from django.db import models
from django.conf import settings
from django.utils.translation import ugettext_lazy as _

from backend.core.tasks import configuracao_cliente_pipeline

import django_rq


class Cliente(models.Model):
    """Model definition for Cliente."""

    # Informações do cliente
    nome = models.CharField(max_length=50)
    subdomain_name = models.CharField(_('Subdomínio'), max_length=50, unique=True)
    email = models.EmailField(max_length=70)
    username = models.CharField(max_length=50)
    password = models.CharField(max_length=50)
    sistema = models.ForeignKey('Sistema', models.PROTECT, related_name='clientes')

    # Informações do droplet
    droplet_id = models.CharField(_('ID do droplet'), max_length=50, blank=True, null=True)
    droplet_ip = models.CharField(_('IP do droplet'), max_length=50, blank=True, null=True)
    droplet_status = models.CharField(max_length=255, blank=True, null=True)
    droplet_logs = models.TextField(blank=True, default='')

    class Meta:
        """Meta definition for Cliente."""
        verbose_name = 'Cliente'
        verbose_name_plural = 'Clientes'

    def __str__(self):
        """Representação da classe na forma de String"""
        return '{} - {}.{}'.format(
            self.nome.title(),
            self.subdomain_name,
            settings.BASE_DOMAIN
        )

    def save(self, *args, **kwargs):
        """
        Sobrescrevendo método save para, logo após
        da criação do Model, executar a task de
        criação do droplet.
        """
        adding = self._state.adding
        super().save(*args, **kwargs)

        if adding:
            self.set_status("Iniciando configuração do cliente.")
            django_rq.enqueue(configuracao_cliente_pipeline, self)

    def set_status(self, text):
        """
        Método para alterar status rapidamente enquanto
        estiver executando tarefas em background

        :param text: Status em que a operação se encontra
        :type text: str
        :returns:
        """
        self.droplet_status = text
        self.append_log(text)
        self.save()

    def append_log(self, log):
        """
        Método para guardar log das operações realizadas
        no droplet do cliente.
        É incrementado em self.droplet_logs.

        :param log: Status em que a operação se encontra
        :type log: str
        :returns:
        """
        self.droplet_logs = "{}\n{}".format(
            self.droplet_logs,
            log
        )
    
    @property
    def user_data(self):
        return self.sistema.droplets_user_data.replace('__domain_name__', self.subdomain_name)
    
    @property
    def domain_name(self):
        return self.sistema.domain_name
