import digitalocean
import psycopg2
import subprocess

from decouple import config

from psycopg2.extensions import ISOLATION_LEVEL_AUTOCOMMIT

from backend.core.models.configuracao import Configuracao

from urllib import parse


def criar_droplet(cliente):
    """
    Função inicial para criação de droplets na digitalocean

    :param cliente: Cliente que foi adicionado.
    :type cliente: model.Cliente.
    :returns: Retorna nada
    :raises: Exception
    """
    if cliente.droplet_id or cliente.droplet_ip:
        raise Exception('Cleinte já possui um droplet.')

    try:
        cliente.set_status("Coletando informações para criação do droplet.")

        configuracoes = Configuracao.get_solo()

        droplet = digitalocean.Droplet(
            token=configuracoes.token,
            name="{}-{}".format(
                cliente.username, cliente.subdomain_name
            ).replace('_', '-'),
            region=configuracoes.droplets_region,
            image=cliente.droplets_image,
            size_slug=cliente.droplets_size,
            backups=cliente.droplets_backups,
            ipv6=cliente.droplets_ipv6,
            user_data=cliente.user_data,
            ssh_keys=configuracoes.ssh_keys
        )

        cliente.set_status("Criando droplet.")
        droplet.create()
    except Exception as err:
        msg = "Erro na criação do droplet. ({})".format(err)
        cliente.set_status(msg)
        raise Exception(msg)

    actions = droplet.get_actions()

    if actions:
        for action in actions:
            if action.type == 'create':
                cliente.set_status("Aguardando disponibilização do droplet.")
                action.wait()
                droplet.load()
                cliente.droplet_ip = droplet.ip_address
                cliente.droplet_id = droplet.id
                cliente.save()
                break
    else:
        error = "Erro. Nenhuma action encontrada."
        cliente.set_status(error)
        raise Exception(error)


def criar_banco(cliente):
    """
    Função para criação do banco de dados do clinente
    no mesmo servidor de bd em que esta aplicação está.

    :param cliente: Cliente que foi adicionado.
    :type cliente: model.Cliente.
    :returns: Retorna nada
    :raises: Exception
    """
    cliente.set_status("Iniciando criação do banco de dados.")
    try:
        database_url = config('DATABASE_URL')
        database_data = parse.urlparse(database_url)

        con = psycopg2.connect(
            dbname='postgres',
            user=database_data.username,
            host=database_data.hostname,
            password=database_data.password
        )

        con.set_isolation_level(ISOLATION_LEVEL_AUTOCOMMIT)

        cur = con.cursor()
        cur.execute('CREATE DATABASE "{}" ;'.format(cliente.subdomain_name))
    except Exception as err:
        msg = "Erro ao criar banco de dados. ({})".format(err)
        cliente.set_status(msg)
        raise Exception(msg)


def criar_subdominio(cliente):
    cliente.set_status("Iniciando criação do subdomínio.")
    process = subprocess.run(
        [
            '/run_add_domain.py',
            cliente.subdomain_name,
            cliente.domain_name,
            cliente.droplet_ip
        ], stdout=subprocess.PIPE
    )
    result = process.stdout.decode('utf-8').strip()
    cliente.append_log(result)
    cliente.set_status(
        result.split('\n').pop()
    )


def configuracao_cliente_pipeline(cliente):
    """
    Pipeline para configuração do servidor do cliente.

    :param cliente: Cliente que foi adicionado.
    :type cliente: model.Cliente.
    :returns: Retorna nada
    :raises: Exception
    """
    try:
        criar_banco(cliente)
        criar_droplet(cliente)
        criar_subdominio(cliente)
    except Exception as err:
        print(err)
        cliente.set_status(
            "Erro ao adicionar novo cliente. ({})".format(err)
        )
